/******************************************************************************\
**
**     COMPANY:	LDW Software
**
**      MODULE:	VillagerManager
**
** DESCRIPTION:	Manages the entire group of villagers
**
**      AUTHOR:	Adrian Francis
**
\******************************************************************************/

#ifndef VILLAGER_MANAGER_H
#define VILLAGER_MANAGER_H

/******************************************************************************\
** LIBRARY INCLUDES
\******************************************************************************/
#include "../Game/Isola.h"
#include "Villager.h"
//#include "Game/theGameState.h"
#include "ldwGameWindow.h"
#include "ldwTypes.h"
#include "../Effects/FloatingAnim.h"
#include "../Tools/DebuggerInterface.h"
#include "../Utils/theStringManager.h"

class thePrizeMsgAssembler;
class ldwSoundTrack;
class ldwSoundFx;
class theGameState;
class ldwGameWindow;
class ldwLog;

/******************************************************************************\
** CONSTANT DEFINITIONS
\******************************************************************************/
const int VillagerManager_c_iMaxVillagers = 150;

#define researchTableX							(821 + randN(180))
#define researchTableY							(630 + randN(3))

#define numTotems								8

#define reaquisitionFreq						10 //how often to reacquire a target (in iterations)
#define targetTolerance							2	//within how many pixels is considered "at target"

enum EAgeSelecter
{
	eChild     = 1,
	eAbleAdult = 2,
	eElderly   = 4,
	eAnyAge    = 7,
	eNotChild  = 6
};

/******************************************************************************\
** CLASS DEFINITIONS
\******************************************************************************/
class CVillagerManager : IDebugger
{
private:

	//class data
	ldwGameWindow *gameWindow;

	ldwLog			*log;
	theStringManager *stringMgr;
	thePrizeMsgAssembler *PrzMsg;

	CVillager villager[VillagerManager_c_iMaxVillagers];

public:

	// Made these public for now (Adrian 08/18/07)
	ldwImage *villagerShadow;

	ldwImageGrid *femaleBodies;
	ldwImageGrid *maleBodies;
	ldwImageGrid *femaleActions;
	ldwImageGrid *maleActions;
	ldwImageGrid *femaleHeads;
	ldwImageGrid *maleHeads;
	ldwImageGrid *femaleHeadsOld;
	ldwImageGrid *maleHeadsOld;

	ldwImage *focusCircle;
	ldwImageGrid *carriedItems;
	ldwImageGrid *		m_pCollectables;
	ldwImageGrid *skelly;
	ldwImageGrid *ripple;
	ldwImageGrid *totems;
	


	//class methods
	//setup
	public:		CVillagerManager();
	public:		void Reset();
	public:		void LoadAssets();
	public:		void ReleaseAssets();
	public:		const bool LoadState(CVillager::SSaveState &sSaveState);
	public:		const bool SaveState(CVillager::SSaveState &sSaveState);

	//process
	public:		void Debug();

	public:		void InitStartingPositions();
	public:		void TakePositions();

	public:		const int SpawnSpecificPeep(const int lName, const ESkill whichSkill, const int howMuchSkill, const EGender whatGender, const int whatAge, const char * fatherName, const char * motherName, const int fatherBodyType, const int fatherHeadType, const int motherBodyType, const int motherHeadType, const int forceX, int const forceY, const bool oldClothes);
	public:		const int SpawnSpecificPeep(const int lName, const ESkill whichSkill, const int howMuchSkill, const EGender whatGender, const int whatAge, const char * fatherName, const char * motherName, const int fatherBodyType, const int fatherHeadType, const int motherBodyType, const int motherHeadType);
	public:		const int SpawnSpecificPeep(const int lName, const ESkill whichSkill, const int howMuchSkill, const EGender whatGender, const int whatAge);
	//public:		const int SpawnSpecificPeep(const int twinOfWhichVillager);
	public:		const int SpawnSpecificPeep(CVillager *twinOfWhichVillager);
	public:		const int SpawnSpecificPeep(CVillager &twinOfWhichVillager);

	public:		void IterateAllVillagers();
	public:		void AllVillagersTimeflowUpkeep();	// TODO: Need a better name!
	public:		void AllVillagersRealtimePhysiologyAndProductivityUpkeep();	// TODO: Need a better name!
	public:		void DrawVillagers();
	public:		void DrawVillager(const int iVillager);

	public:		void SetChief(CVillager &newChief);
	public:		const bool ChiefExists();
	public:		const int ChiefPredisposition();
	public:		const bool ChanceOfChiefBoost(int preDisposition);
			
 	private:	void DrawCarriedItems(const CVillager &Villager, const ldwPoint sPos);

	public:		const int SelectRandomLivingVillager(const bool adultNeeded);
	public:		const int SelectRandomLivingVillager(const bool adultNeeded, const int gender);
	public:		const int SelectRandomLivingChild(void);
	public:		const int SelectRandomLivingVillager(const int ageCutoff, const int gender);
	public:		const int SelectOtherAvailableMatingVillager(CVillager &villagerToOmit);
	public:		const int SelectPlayMateFor(CVillager &Villager);
	public:		const bool SelectChosenOne();

	public:		const int SelectOtherSickVillager(CVillager &VillagerToOmit);

	public:		const int CheckForVillagerClick(const ldwPoint sClickPos);
	public:		CVillager *CheckForOtherWaitingVillagerAtPoint(const CVillager *pWhoIsChecking, const ldwPoint click);
	
	public:		void MakeInFocus(const int whichVillager);
	public:		void MakeInFocus(const CVillager *pVillager);
	public:		void SetNoFocus();	
	public:		const ldwPoint GetVillagerPosition(const int whichVillager) { return ldwPoint(villager[whichVillager].XPos(), villager[whichVillager].YPos()); }
	
	public:		const int FindCorpse();
	public:		void InvokeCelebration(int whichEvent);

	public:		void MakeRandomVillagersSick(const int chance, const bool injury = false, const int injuryAmt = 0);
	public:		void MakeRandomVillagersDead(const int chance, EGender gender);
	public:		void MakeRandomVillagersVanish(const int chance, EGender gender);
	public:		void CureAllVillagers();
	public:		void MakeAllVillagersForgetWhatToDo(bool forgetPlans);
	public:	    void MakeRandomVillagersInjured(const int chance, const int baseinjuryAmt, const int randomAmt);
	public:		void ImproveRandomVillagersSkill(const int chance, const ESkill whichSkill, const int baseAmt, const int randomBonus, const bool needChild);
	public:		void ChangeRandomVillagersClothes(const int chance, const bool needChild);
	public:	    void AddRandomVillagerLike(const int chance, ELike eLike);
	public:	    void AddRandomVillagerDislike(const int chance, ELike eDislike);
	public:		void RandomSkillPenalty(CVillager *pVillager, int baseAmount, int randomAmount);
	public:		void RandomSkillBonus(CVillager *pVillager, int baseAmount, int randomAmount);
	public:		void MakeAllVillagersDo(const EBehavior eBehavior, EAgeSelecter ageSelecter, EGender gender, bool chiefOk, int *candidateCount);
	public:		void MakeAllVillagersDo(const EBehavior eBehavior, EAgeSelecter ageSelecter, EGender gender, bool chiefOk);
	public:		void MakeAllVillagersDo(const EBehavior eBehavior, int age,EGender gender, bool chiefOk, int *candidateCount);
	public:		void MakeAllVillagersDo(const EBehavior eBehavior, int age,EGender gender, bool chiefOk);
	public:		void MakeSomeVillagersDo(const EBehavior eBehavior, int chance, EAgeSelecter ageSelecter,EGender gender,bool chiefOk);
	public:		void TeleportAllVillagersTo(int xPos, int yPos, EAgeSelecter ageSelecter, EGender gender, bool chiefOk);
	public:		void TeleportAllVillagersTo(int xPos, int yPos, EAgeSelecter ageSelecter, EGender gender, bool chiefOk, int *candidateCount);
	public: 	void TeleportAllVillagersTo(int xPos, int xRand, int yPos, int yRand, EAgeSelecter ageSelecter, EGender gender, bool chiefOk, int *candidateCount);
	public:		CVillager *GetRandomVillagerWithSkill(ESkill eSkill, int skillAmount, EAgeSelecter ageSelecter,EGender gender,bool chiefOk,int *candidateCount);
	public:		const EGender ExtremeLimitGenderCorrect();
	public:		void GatherChildrenForStory();
	public:		CVillager *GetChief();

	//access
	public:		CVillager &GetVillager(const int iVillager);
	public:		CVillager *GetVillagerInFocus();
	public:		CVillager *GetVillagerAt(const ldwPoint sPos, const int iExceptFor, const bool bIncludeCorpses = false);
	public:		CVillager *GetVillagerWaitingAt(const ldwPoint sPos, const CVillager *pWhoIsChecking);
	public:		CVillager *GetRandomVillager(EAgeSelecter ageSelecter,EGender gender = eGender_Unknown,bool chiefOk = true,int *candidateCount = 0);
	public:		const bool IsRoomToPopulate();
	public:		const int Population();
	public:		const int Totems();
	public:		const int GenderCounts(int &iMaleCount, int &iFemaleCount);
	public:		const int MothersCaringForBabies();
	public:		const bool HasAnyVillagerScaredBirds();
	public:		const bool IsAnyoneCarrying(const ECarrying eCarrying) const;
	public:		CVillager *GetVillagerDoing(const EBehavior eBehavior, bool isRandom, int *candidateCount);
	public:		CVillager *GetVillagerDoing(const EBehavior eBehavior);
	public:		void ChangeAllBehavior(const EBehavior targetBehavior, const EBehavior newBehavior);
	
	public:		bool VillagerExists(const int iVillager);

};

/******************************************************************************\
** GLOBAL DECLARATIONS
\******************************************************************************/
extern CVillagerManager		VillagerManager;

#endif

/******************************************************************************\
** END OF FILE
\******************************************************************************/